// GLCamera.cpp: implementation of the GLCamera class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GLCamera.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

/*
 * 3D Camera class. Encapsulates OpenGL camera control for both orbiting and moving camera.
 * Author: Ryan Holmes
 * E-mail: Ryan <dot> Holmes <at> asu <dot> edu
 */

GLCamera::GLCamera()
{
	nearPlane = 1.0;
	farPlane = 500.0;
	fromPoint.x = 10;
	upVector.y = 1;
	viewAngle = 45;
	viewWidth = 1.0;
	viewHeight = 1.0;
	aspect = 1.0;
	speed = 0.05;
}

GLCamera::~GLCamera()
{

}

void GLCamera::setGLView() {

	::glMatrixMode(GL_PROJECTION);
	::glLoadIdentity();
	::gluPerspective( viewAngle, aspect, nearPlane, farPlane);
	::glMatrixMode(GL_MODELVIEW);
	setGLViewModel();
}

void GLCamera::setGLViewModel() {

	::glLoadIdentity();
	::gluLookAt(fromPoint.x,
		        fromPoint.y,
				fromPoint.z,
				atPoint.x,
				atPoint.y,
				atPoint.z,
				upVector.x,
				upVector.y,
				upVector.z);
}

void GLCamera::setViewport(int x, int y, int width, int height)
{
	::glViewport(x, y, width, height);
	viewWidth = (GLdouble) width;
	viewHeight = (GLdouble) height;
	aspect = viewWidth / viewHeight;
}

void GLCamera::zoom(double amount)
{
	Vector reverseLoS = fromPoint - atPoint;

	if (amount > 0) {				// If positive, zoom out.
		reverseLoS.scaleBy(1.1);
	} else if (amount < 0) {		// Otherwise, zoom in
		if (reverseLoS.getLength() > 0.05) { // Prevent the from point from hitting the at point.
			reverseLoS.scaleBy(.9);
		}
	}
	fromPoint = reverseLoS + atPoint;
}


void GLCamera::pan(double downDegrees, double rightDegrees)
{
	// Find the vector that defines the point to move.
	Vector reverseLoS = fromPoint - atPoint;

	// Find the orthogonal local scheme to use.
	Vector right = upVector.Cross(reverseLoS);
	Vector orthogUp = reverseLoS.Cross(right);

	// Rotate both vectors in question around those axes
	// we just found.
	reverseLoS.rotateAxis(right, downDegrees);
	reverseLoS.rotateAxis(orthogUp, rightDegrees);
	upVector.rotateAxis(right, downDegrees);
	upVector.rotateAxis(orthogUp, rightDegrees);

	// Store the vectors back into our variables. We
	// translate the reversed line of sight to return
	// it to a point. The up vector is just a vector,
	// so needs no translation.
	fromPoint = reverseLoS + atPoint;
}

void GLCamera::turn(double downDegrees, double rightDegrees)
{
	// Find the vector that defines the point to move.
	Vector LoS = atPoint - fromPoint;

	// Find the orthogonal local scheme to use.
	Vector right = LoS.Cross(upVector);
	Vector orthogUp = LoS.Cross(right);

	// Rotate both vectors in question around those axes
	// we just found.
	LoS.rotateAxis(right, downDegrees);
	LoS.rotateAxis(orthogUp, rightDegrees);
	upVector.rotateAxis(right, downDegrees);
	upVector.rotateAxis(orthogUp, rightDegrees);

	// Store the vectors back into our variables. We
	// translate the line of sight to return
	// it to a point. The up vector is just a vector,
	// so needs no translation.
	atPoint = LoS + fromPoint;
}

void GLCamera::changeSpeed(double percent)
{
	speed = speed * percent;
	speed = fabs(speed);
}

void GLCamera::setSpeed(double newSpeed)
{
	speed = fabs(newSpeed);
}

void GLCamera::move(bool forward)
{
	Vector offset = atPoint - fromPoint;
	if (forward) {
		offset.scaleBy(speed);
	} else {
		offset.scaleBy(-speed);
	}

	fromPoint.translateBy(offset);
	atPoint.translateBy(offset);
}
