// OpenGLSkeletonView.cpp : implementation of the COpenGLSkeletonView class
//

#include "stdafx.h"
#include "OpenGLSkeleton.h"

#include "OpenGLSkeletonDoc.h"
#include "OpenGLSkeletonView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// COpenGLSkeletonView

IMPLEMENT_DYNCREATE(COpenGLSkeletonView, CView)

BEGIN_MESSAGE_MAP(COpenGLSkeletonView, CView)
	//{{AFX_MSG_MAP(COpenGLSkeletonView)
	ON_WM_CREATE()
	ON_WM_DESTROY()
	ON_WM_ERASEBKGND()
	ON_WM_SIZE()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_MOUSEMOVE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// COpenGLSkeletonView construction/destruction

COpenGLSkeletonView::COpenGLSkeletonView()
{
	// The closest depth that will be displayed is 1.0 in front
	// of us. The near plane can never be 0 or negative, or OpenGL will
	// error.
	m_near_plane = 1.0f;
	// The furthest away depth will be 50.0 in front of us.
	m_far_plane = 50.0f;
	// bgColor is a COLORREF. This is a packed int, where each
	// 8 bits in a particular order store red, green, and blue
	// information. RGB is a macro that packs three numbers
	// between 0 and 255 into a COLORREF. The inverse can be
	// done with the GetRValue, GetGValue, and GetBValue macros.
	bgColor = RGB(0, 0, 0);

	lMouseDown = false;
	rMouseDown = false;
}

COpenGLSkeletonView::~COpenGLSkeletonView()
{
}

BOOL COpenGLSkeletonView::PreCreateWindow(CREATESTRUCT& cs)
{
	// Tell the window-creation routines that we want this
	// window to clip children and siblings.
	cs.style |= WS_CLIPCHILDREN | WS_CLIPSIBLINGS;

	// Allow the window-creation routines to do their work.
	return CView::PreCreateWindow(cs);
}

/////////////////////////////////////////////////////////////////////////////
// COpenGLSkeletonView drawing

void COpenGLSkeletonView::OnDraw(CDC* pDC)
{
	// Get a pointer to the Doc
	COpenGLSkeletonDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// Clear the window with the bgcolor, and set the
	// depth buffer to the farthest away depth.
	::glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Get the display list the Doc has built, and
	// send it to OpenGL to render.
	::glCallList(pDoc->getList());

	// Make sure that OpenGL has completely finished drawing
	// before returning control to the Windows drawing routines.
	// This is overkill, and slows things down.
	::glFinish();
	::glFlush();
	// Get a pointer to the Device Context we're rendering into.
	::HDC hDC = ::wglGetCurrentDC();
	// Swap the buffer we just drew into with the one being
	// displayed. This means our picture appears all at once,
	// instead of being drawn on the screen one piece at a time.
	::SwapBuffers(hDC);
}

/////////////////////////////////////////////////////////////////////////////
// COpenGLSkeletonView diagnostics

#ifdef _DEBUG
void COpenGLSkeletonView::AssertValid() const
{
	CView::AssertValid();
}

void COpenGLSkeletonView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

COpenGLSkeletonDoc* COpenGLSkeletonView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(COpenGLSkeletonDoc)));
	return (COpenGLSkeletonDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// COpenGLSkeletonView message handlers

int COpenGLSkeletonView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;
	
    PIXELFORMATDESCRIPTOR pfd = {
	sizeof(PIXELFORMATDESCRIPTOR),
		1,						// Version number
		PFD_DRAW_TO_WINDOW |	// Draw in window
		PFD_SUPPORT_OPENGL |	// Use openGL
		PFD_DOUBLEBUFFER,		// Double-buffered
		PFD_TYPE_RGBA,			// RGB and Alpha
		24,						// 24-bit color depth
		0, 0, 0, 0, 0, 0,		// color bits ignored
		0,						// no alpha buffer
		0,						// shift bit ignored
		0,						// no accumulation buffer
		0, 0, 0, 0,				// accum bits ignored
		32,						// 32-bit z-buffer
		0,						// no stencil buffer
		0,						// no auxiliary buffer
		PFD_MAIN_PLANE,			// main layer
		0,						// reserved
		0, 0, 0					// layer masks ignored
	};

	m_pDC = new CClientDC(this);
	ASSERT(m_pDC != NULL);

	int pixelformat = ::ChoosePixelFormat(m_pDC->GetSafeHdc(), &pfd);
	if (pixelformat == 0) {
		AfxMessageBox("Bad Format Choose");
	}

	BOOL success = ::SetPixelFormat(m_pDC->GetSafeHdc(), pixelformat, &pfd);
	if (!success) {
		AfxMessageBox("Bad Pixel Format");
	}

	int n = ::GetPixelFormat(m_pDC->GetSafeHdc());

	::DescribePixelFormat(m_pDC->GetSafeHdc(), n, sizeof(pfd), &pfd);
	
	m_hRC = ::wglCreateContext(m_pDC->GetSafeHdc());
	::wglMakeCurrent(m_pDC->GetSafeHdc(), m_hRC);
	
	return 0;
}

void COpenGLSkeletonView::OnDestroy() 
{
	CView::OnDestroy();
	
	// Get the GL context, turn off GL, and delete
	// the context we retrieved.
	HGLRC hrc = ::wglGetCurrentContext();
	::wglMakeCurrent(NULL, NULL);
	if (hrc != NULL) {
		::wglDeleteContext(hrc);
	}
	if (m_pDC != NULL) {
		delete m_pDC;
	}
}

BOOL COpenGLSkeletonView::OnEraseBkgnd(CDC* pDC) 
{
	// Returning true here tells MFC that background
	// has been erased. This is a lie, but OpenGL will
	// do the erasing in the appropriate place itself.
	// If MFC does it, you get a white flicker when you
	// resize the window.
	return true;
}

void COpenGLSkeletonView::OnSize(UINT nType, int cx, int cy) 
{
	CView::OnSize(nType, cx, cy);
	
	camera.setViewport(0, 0, cx, cy);
	camera.setGLView();
}

void COpenGLSkeletonView::OnInitialUpdate() 
{
	CView::OnInitialUpdate();
	
	// Get a pointer to the Doc object.
	COpenGLSkeletonDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// Now we do OpenGL setup
	GetClientRect(&m_window_size);
	if (m_window_size.bottom < 1) {
		m_window_size.bottom = 1;
	}

	camera.fromPoint = Vector(-1, 2, 2.5);
	camera.atPoint.zero();
	camera.upVector = Vector(0, 1, 0);

	camera.setGLView();

	// Set the clear depth to be the furthest away depth
	// it can handle. This is unnecessary, because the
	// default value is 1.0. It is done here as an example only.
	::glClearDepth(1.0f);
	// Tell OpenGL to use the depth buffer (z-buffer)
	::glEnable(GL_DEPTH_TEST);
	// Tell OpenGL to use bgColor for clearing the window.
	::glClearColor(GetRValue(bgColor) / 255.0f,
		           GetGValue(bgColor) / 255.0f,
				   GetBValue(bgColor) / 255.0f,
				   1.0f);
	// Tell OpenGL to display both sides of polygons, instead
	// of just the "outside" or "inside" ones.
	::glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);

	// Set lighting colors.
	GLfloat d[] = {1.0f, 1.0f, 1.0f, 1.0f};
	GLfloat a[] = {0.3f, 0.3f, 0.3f, 1.0f};
	GLfloat s[] = {1.0f, 1.0f, 1.0f, 1.0f};
	::glLightfv(GL_LIGHT0, GL_DIFFUSE, d);
	::glLightfv(GL_LIGHT0, GL_AMBIENT, a);
	::glLightfv(GL_LIGHT0, GL_SPECULAR, s);

	// Set light0 and turn it on.
	GLfloat p[] = {3.0f, 3.0f, 3.0f, 1.0f};
	::glLightfv(GL_LIGHT0, GL_POSITION, p);
    ::glEnable(GL_LIGHT0);

	// Set shading to flat shading.
	::glShadeModel(GL_FLAT);
	// Turn on lighting
	::glEnable(GL_LIGHTING);
	// Tell OpenGL to fill polygons (Not draw them as wire
	// frame, etc.)
	::glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

	// Set the outside color to be blue and the inside to be red.
	GLfloat outside[] = {0, 0, 0.7f, 1.0f};
	GLfloat inside[] = {0.8f, 0, 0, 1.0f};
	::glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, outside);
	::glMaterialfv(GL_BACK, GL_AMBIENT_AND_DIFFUSE, inside);

	// Set the highlight color for both sides to a light
	// grey, and medium-sized.
	GLfloat specular[] = {0.7f, 0.7f, 0.7f, 1.0f};
	::glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
	::glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 70.0);

	// Tell the Doc to build a new list. This list contains all of
	// the OpenGL calls necessary to display the geometry drawn
	// by this program. More complex programs could have many lists,
	// or could do some of the drawing directly and some in lists.
	pDoc->buildList();

	// Tell MFC to call OnDraw when the opportunity presents itself.
	Invalidate(FALSE);	
}

/*
void COpenGLSkeletonView::setGLView() {

	// Tell OpenGL we want to change the projection matrix
	::glMatrixMode(GL_PROJECTION);
	// Load the identity matrix into the projection matrix
	::glLoadIdentity();
	// Calculate an aspect ratio based on the size of the window (width / height)
	aspect = (GLfloat) m_window_size.right /
		     (GLfloat) m_window_size.bottom;
	// Load the projection matrix with a perspective projection.
	::gluPerspective( 45, aspect, m_near_plane, m_far_plane);
	// Tell OpenGL we want to change the modelview matrix
	::glMatrixMode(GL_MODELVIEW);
	// Clear any previous transformations.
	::glLoadIdentity();
	// Load a transformation that positions the camera such that
	// it is looking from the point (-1, 2, 2.5), toward the point
	// (0,0,0), and up is in the direction (0,1,0).
	::gluLookAt(-1, 2, 2.5,
		        0, 0, 0,
				0, 1, 0);
}
*/

void COpenGLSkeletonView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	lMouseDown = true;
	lastClickPoint = point;
	
	CView::OnLButtonDown(nFlags, point);
}

void COpenGLSkeletonView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	lMouseDown = false;
	
	CView::OnLButtonUp(nFlags, point);
}

void COpenGLSkeletonView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	rMouseDown = true;
	lastClickPoint = point;
	
	CView::OnRButtonDown(nFlags, point);
}

void COpenGLSkeletonView::OnRButtonUp(UINT nFlags, CPoint point) 
{
	rMouseDown = false;
	
	CView::OnRButtonUp(nFlags, point);
}

void COpenGLSkeletonView::OnMouseMove(UINT nFlags, CPoint point) 
{
	bool redraw = false;
	bool resizeVolume = false;
	// Positive deltaX means mouse moved right.
	int deltaX = point.x - lastClickPoint.x;
	// Positive deltaY means mouse moved down.
	int deltaY = point.y - lastClickPoint.y;

	if (lMouseDown && rMouseDown) {
		// Avoid the case where both buttons are down.
	} else if (lMouseDown) {
		// The left mouse button is down, so we pan.
		// Note that we use the negative of the deltas. If
		// the mouse has moved to the right, we want to move
		// our viewpoint around to the left. This makes the
		// object look like it has turned to the right. Similarly
		// for up and down.
		camera.pan(-deltaY, -deltaX);
		redraw = true;
	} else if (rMouseDown) {
		camera.zoom(deltaY);
		redraw = true;
		resizeVolume = true;
	}

	if (redraw) {
		if (resizeVolume) {
//			updateNearFar();
		}
		camera.setGLView();
		Invalidate(FALSE);
	}

	// Save the current point for the next call.
	lastClickPoint = point;
	
	CView::OnMouseMove(nFlags, point);
}
